{ **********************************************************************
  *                        Program FITEXPO.PAS                         *
  *                            Version 1.2                             *
  *                     (c) J. Debord, July 1997                       *
  **********************************************************************
  This program performs a nonlinear least squares fit of a sum of
  decreasing exponentials :

            y = Ymin + A.exp(-a.x) + B.exp(-b.x) + C.exp(-c.x)

  The following parameters are passed on the command line :

    1st parameter = Name of input file (default extension = .DAT)
                    The structure of the input file is described in
                    REG_IN.INC
    2nd parameter = Number of exponentials (1 to 3, default = 1)
    3rd parameter = 1 if the function includes a constant term (Ymin)

  There are 3 examples taken from the book of Gibaldi and Perrier
  (M. GIBALDI & D. PERRIER, Pharmacokinetics, 2nd edition, Dekker 1982) :

    IV2.DAT   (2 exponentials)
    ORAL1.DAT (2 exponentials)
    ORAL2.DAT (3 exponentials)

  The program may be executed from Turbo Pascal's integrated environment,
  in which case the parameters (e.g. IV2 2) are entered through the
  "Parameters" option of the menu, or from DOS (after compilation into
  an executable file), in which case the parameters are entered on the
  command line (e.g. FITEXPO IV2 2).
  ********************************************************************** }

program FitExpo;

uses
  Crt, Graph, FMath, Matrices, Optim,
  Polynom, Stat, Regress, PaString, Plot;

const
  MAX_EXP      = 3;       { Maximum number of exponentials }
  NO_REAL_ROOT = - 2;     { No real exponent }
  ITMARQ       = 500;     { Number of iterations allowed to Marquardt }
  TOLMARQ      = 1.0E-4;  { Required precision for Marquardt estimation }

var
  InFName  : String;   { Name of input file }
  Title    : String;   { Title of study }
  XName,
  YName    : String;   { Variable names }
  N        : Integer;  { Number of points }
  X, Y     : PVector;  { Point coordinates }
  Ycalc    : PVector;  { Expected Y values }
  W        : PVector;  { Weights }
  N_exp    : Integer;  { Number of exponentials }
  ConsTerm : Boolean;  { Flags the presence of a constant term Ymin }
  Lbound,
  Ubound   : Integer;  { Indices of first & last fitted parameters }
  B        : PVector;  { Regression parameters }
  V        : PMatrix;  { Variance-covariance matrix of parameters }
  ErrCode  : Integer;  { Error code }

{ ----------------------------------------------------------------------
  Define here the function used to compute the variance of an observed
  y value. The true variance will be : V(y) = Vr * VarFunc(y), where Vr
  is the residual variance (estimated by the program).

  Ex. : VarFunc(y) = Sqr(y)  for a variance proportional to y^2

  For unweighted regression, simply set VarFunc to 1.0
  ---------------------------------------------------------------------- }

  function VarFunc(Y : Float) : Float;
  begin
    { Here we assume a constant variance }
    VarFunc := 1.0;
  end;

  procedure ReadCmdLine;
  { Reads command line parameters }
  var
    I, ErrCode : Integer;
  begin
    { Name of input file }
    InFName := ParamStr(1);
    if Pos('.', InFName) = 0 then InFName := InFName + '.DAT';

    { Number of exponentials }
    N_exp := 0;
    Val(ParamStr(2), N_exp, ErrCode);
    if (ErrCode <> 0) or (N_exp < 1) then N_exp := 1;
    if N_exp > MAX_EXP then N_exp := MAX_EXP;

    { Presence of constant term }
    I := 0;
    Val(ParamStr(3), I, ErrCode);
    ConsTerm := (I = 1);
  end;

  function FuncName : String;
  { Returns the name of the regression function }
  var
    Name : String;
  begin
    Name := 'y = ';
    if ConsTerm then
      Name := Name + 'Ymin + ';
    Name := Name + 'A.exp(-a.x)';
    if N_exp > 1 then
      Name := Name + ' + B.exp(-b.x)';
    if N_exp > 2 then
      Name := Name + ' + C.exp(-c.x)';
    FuncName := Name;
  end;

  function FirstParam : Integer;
  { Returns the index of the first parameter to be fitted }
  begin
    if ConsTerm then
      FirstParam := 0
    else
      FirstParam := 1;
  end;

  function LastParam : Integer;
  { Returns the index of the last parameter to be fitted }
  begin
    LastParam := 2 * N_exp;
  end;

  function ParamName(I : Integer) : String;
  { Returns the name of the I-th parameter }
  begin
    case I of
      0 : ParamName := 'Ymin';
      1 : ParamName := 'A   ';
      2 : ParamName := 'a   ';
      3 : ParamName := 'B   ';
      4 : ParamName := 'b   ';
      5 : ParamName := 'C   ';
      6 : ParamName := 'c   ';
    end;
  end;

  function RegFunc(X : Float; B : PVector) : Float;
  { --------------------------------------------------------------------
    Computes the regression function
    B is the vector of parameters, such that :
      B^[0] = Ymin     B^[1] = A     B^[2] = a
                       B^[3] = B     B^[4] = b
                       B^[5] = C     B^[6] = c
    -------------------------------------------------------------------- }
  var
    I : Integer;
    S : Float;
  begin
    if ConsTerm then
      S := B^[0]
    else
      S := 0.0;
    for I := 1 to N_exp do
      S := S + B^[2 * I - 1] * Exp(- B^[2 * I] * X);
    RegFunc := S;
  end;

  procedure DerivProc(X : Float; B, D : PVector);
  { --------------------------------------------------------------------
    Computes the derivatives of the regression function with respect to
    the parameters, for a given data point
    Input  : X = point abscissa
             B = regression parameters
    Output : D = vector of derivatives, d(i) = df(x, b) / db(i)
    -------------------------------------------------------------------- }
  var
    I, P, Q : Integer;
    E : Float;
  begin
    D^[0] := 1.0;
    for I := 1 to N_exp do
      begin
        Q := 2 * I;
        P := Pred(Q);
        E := Exp(- B^[Q] * X);
        D^[P] := E;
        D^[Q] := - X * B^[P] * E;
      end;
  end;

  function ApproxFit : Integer;
  { --------------------------------------------------------------------
    Approximate fit of the sum of exponentials by linear regression
    Ref. : R. GOMENI & C. GOMENI, Automod : A polyalgorithm for an
           integrated analysis of linear pharmacokinetic models
           Comput. Biol. Med., 1979, 9, 39-48
    -------------------------------------------------------------------- }
  var
    I, K, M : Integer;
    X1, Y1  : PVector;  { Scaled coordinates }
    W1      : PVector;  { Weights }
    U       : PMatrix;  { Variables for linear regression }
    P       : PVector;  { Linear regression parameters }
    C, Z    : PVector;  { Coefficients and roots of polynomial }
    V       : PMatrix;  { Variance-covariance matrix }
    H       : Float;
  begin
    M := Pred(2 * N_exp);
    DimVector(X1, N);
    DimVector(Y1, N);
    DimVector(W1, N);
    DimMatrix(U, M, N);
    DimMatrix(V, M, M);
    DimVector(P, M);
    DimVector(C, N_exp);
    DimVector(Z, N_exp);
    CopyVector(X1, X, 1, N);
    CopyVector(Y1, Y, 1, N);

    { Change scale so that the X's begin at zero }
    if X^[1] <> 0.0 then
      for K := 1 to N do
        X1^[K] := X1^[K] - X^[1];

    { Compute weights }
    for K := 1 to N do
      W1^[K] := 1.0 / VarFunc(Y1^[K]);

    { Estimate constant term at 90% of the lowest observed value }
    if ConsTerm then
      begin
        B^[0] := 0.9 * Min(Y1, 1, N);
        for K := 1 to N do
          Y1^[K] := Y1^[K] - B^[0];
      end;

    { ------------------------------------------------------------------
      Perform linear regression on the linearized form of the equation :

      y = p(0) + p(1) * x + p(2) * x^2 + ... + p(N_exp-1) * x^(N_exp-1)

                    (x                          (x    (x
         + p(N_exp) | y dx + ... + p(2*N_exp-1) | ....| y dx
                    )0                          )0    )0
      ------------------------------------------------------------------ }

    { Compute increasing powers of X }
    if N_exp > 1 then
      for K := 2 to N do
        begin
          U^[1]^[K] := X1^[K];
          for I := 2 to Pred(N_exp) do
            U^[I]^[K] := U^[I - 1]^[K] * X1^[K];
        end;

    { Compute integrals by trapezoidal rule }
    for K := 2 to N do
      begin
        H := 0.5 * (X1^[K] - X1^[K - 1]);
        U^[N_exp]^[K] := U^[N_exp]^[K - 1] + (Y1^[K] + Y1^[K - 1]) * H;
        for I := Succ(N_exp) to M do
          U^[I]^[K] := U^[I]^[K - 1] + (U^[I - 1]^[K] + U^[I - 1]^[K - 1]) * H;
      end;

    { Fit the linearized equation }
    if WMulFit(U, Y1, W1, N, M, True, P, V) = MAT_SINGUL then
      ApproxFit := MAT_SINGUL
    else
      begin
      { ----------------------------------------------------------------
        The exponents are the real roots of the polynomial :
        x^N_exp + p(N_exp) * x^(N_exp-1) - p(N_exp+1) * x^(N_exp-2) +...
        ---------------------------------------------------------------- }

        { Compute coefficients of polynomial }
        C^[N_exp] := 1.0;
        for I := 1 to N_exp do
          if Odd(I) then
            C^[N_exp - I] := P^[N_exp + I - 1]
          else
            C^[N_exp - I] := - P^[N_exp + I - 1];

        { Solve polynomial }
        if RootPol3(C, N_exp, Z) <> N_exp then
          ApproxFit := NO_REAL_ROOT
        else
          begin
            { Sort exponents in decreasing order }
            DQSort(Z, 1, N_exp);

            { Compute coefficients of exponentials by weighted
              linear regression on the exponential terms }
            for I := 1 to N_exp do
              for K := 1 to N do
                U^[I]^[K] := Exp(- Z^[I] * X1^[K]);
            if WMulFit(U, Y1, W1, N, N_exp, False, P, V) = MAT_SINGUL then
              ApproxFit := MAT_SINGUL
            else
              begin
                { Get model parameters }
                for I := 1 to N_exp do
                  begin
                    { Correct for scale change if necessary }
                    if X^[1] <> 0.0 then
                      P^[I] := P^[I] * Exp(Z^[I] * X^[1]);

                    { Get coefficients and exponents }
                    B^[2 * I - 1] := P^[I];  { Coefficients }
                    B^[2 * I] := Z^[I];      { Exponents }
                  end;
                ApproxFit := MAT_OK;
              end;
          end;
      end;

    DelVector(X1, N);
    DelVector(Y1, N);
    DelVector(W1, N);
    DelMatrix(U, M, N);
    DelMatrix(V, M, M);
    DelVector(P, M);
    DelVector(C, N_exp);
    DelVector(Z, N_exp);
  end;

  {$F+}
  function PlotRegFunc(X : Float) : Float;
  { Defines the function to be plotted }
  begin
    PlotRegFunc := RegFunc(X, B);
  end;
  {$F-}

  {$I REG_IN.INC}    { Read input file }

  {$I REG_NL.INC}    { Nonlinear regression }

  {$I REG_OUT.INC}   { Write output file }

  {$I REG_PLOT.INC}  { Plot function }

{ *************************** Main program ***************************** }

begin
  ReadCmdLine;
  ReadInputFile;
  if ErrCode = 0 then
    FitModel;
  if ErrCode = MAT_OK then
    begin
      { Path to the graphic drivers (Default = C:\BP\BGI) }
      { BGIPath := 'C:\BP\BGI'; }
      GraphTitle.Text := Title;
      PlotFuncAddr := @PlotRegFunc;
      YAxis.Scale := LOG_SCALE;
      PlotGraph;
      WriteOutputFile;
    end;
end.
