	title	PostScript device driver

;;;	post17.sys - PostScript device driver
;
;	P. Jarvis              16/02/1993
;
;
;	This driver replaces LPT3: and processes any
;	text sent to it. The text is converted to
;	PostScript which is then output to either LPT1:
;	or LPT2:.
;
;	This driver is based on `postprn', a free
;	driver by Russ Nelson (nelson@clutx.clarkson.edu)

;;	post17.sys - Compilation
;
;	The source compiles under Microsoft MASM 6.0 as
;	follows:
;		ml post17.asm
;		link ps
;		exe2bin post17.exe post17.sys

;;	post17.sys - Installation
;
;	Put the post17.sys file in a suitable directory and
;	add the following line to your config.sys file.
;
;	device=c:\post17.sys
;
;	If the device driver is not in the root directory
;	then the full path must be specified in the config.sys
;	entry.
;
;	post17.sys can take two parameters as follows:
;
;	/W     forces output to wrap after column 80
;	/P     specifies the parallel port to use. This
;	       may be /P1 - for LPT1: - or /P2 - for LPT2:.
;	       The default being LPT1:.

;;	Symbol definitions.
;
CTRL_D	     equ	04H
EXIT_BUSY    equ	0300H
EXIT_ERROR   equ	8100H
EXIT_SUCCESS equ	0100H
EXIT_UNKNOWN equ	8103H

;;	Structure definitions.
;
uptr	STRUCT			;  dword format
offs	dw	?		;    Offset
segm	dw	?		;    Segment
uptr	ENDS


req	STRUCT			;  Request block
hdrlen	db	?		;    Header length
unit	db	?		;    Unit number for request
code	db	?		;    Command code
status	dw	?		;    Returned status
reserv	db	8 dup (?)	;    8 reserved bytes
media	db	?		;    Media description byte
adr	dd	?		;    Memory address for transfer
count	dw	?		;    Byte or sector count
sector	dw	?		;    Starting sector
req	ENDS

cseg	segment	byte
	assume	cs:cseg
	org	0

;;	Device Header.
;
	dw	-1		;  Offset to next driver
	dw	-1		;  Segment of next driver
	dw	8800H		;  Attributes (Character and open/close)
	dw	strat		;  Offset to stategy routine
	dw	intr		;  Offset to interrupt routine
        db	'LPT3    '      ;  Device name

;;	Command branch table.
;
 cmds	label	word
	dw	init		;  Initialise
	dw	done		;  Media clock (block devices only)
	dw	done		;  Build parameter block (ignored)
	dw	unknown		;  I/O Control read
	dw	done		;  Read
	dw	busy		;  Non-destructive read
	dw	done		;  Input status
	dw	done		;  Flush input buffers
	dw	writec		;  Write
	dw	writec		;  Write with verify
	dw	writes		;  Output status
	dw	unknown		;  Flush output buffers
	dw	unknown		;  I/O Control write
	dw	open		;  Device open
	dw	close		;  Device close
	dw	done		;  Removable media
	dw	done		;  Output until busy

;;	Variables.
;
printer	dw	0		;  Printer ordinal (0 = lpt1:)
auto_wrap db	0		;  Wrap column number if not zero
opened	db	0		;  Output file is to be opened
request	dd	?		;  Address of request block

;;;	Command processors.
;
busy:
	mov	ax,EXIT_BUSY
	ret

close:
	cmp	cs:opened,1	;  Check if just opened
	je	close2		;  Nope
	mov	dx,cs:printer	;  (dx) = printer ordinal
	mov	al,CTRL_D	;  (al) = End of job code
	call	outchar
close2:
	ret

open:
	mov	cs:opened,1	;  Flag output to be opened
	mov	ax,EXIT_SUCCESS
	ret

done:
	mov	ax,EXIT_SUCCESS
	ret

unknown:
	mov	ax,EXIT_UNKNOWN
	ret

writec:
	cmp	cs:opened,0	;  Check if first output
	je	writec2		;  Nope
	mov	cs:opened,0	;  Clear just opened flag
	call	prolog
writec2:
	les	bx,cs:request	;  (es:bx) = Address of request block
	mov	cx,es:[bx].req.count;  (cx) = Byte count
	lds	si,es:[bx].req.adr  ;  (ds:si) = address of data to write
	mov	dx,cs:printer	;  (dx) = Printer port
writec4:
	lodsb			;  (al) = Next character
	call	outchar
	jc	writec6		;  Error
	loop	writec4
	ret
writec6:
	les	bx,cs:request	;  (es:bx) = Address of request block
	sub	es:[bx].req.count,cx;  Allow for characters printers
	ret

writes:				;  Write output status
	mov	dx,cs:printer	;  (dx) = Printer ordinal
	mov	ah,2
	int	17H		;  Status printer
	call	status		;  Check return status
	jc	writes2		;  Error
	test	ah,80H		;  Check if busy
	jz	writes4		;  Yep
	mov	ax,EXIT_SUCCESS
	ret
writes2:
	mov	ah,80H		;  Flag problem
	ret
writes4:
	mov	ax,EXIT_BUSY
	ret


;;	prolog - Write PostScript prolog.
;
;	Entry	None.
;
;	Exit	None.

prolog:
	mov	si,offset prolog_a
	call	prolog4
	cmp	auto_wrap,0	;  Check for wrap 
	je	prolog2		;  No wrap
	mov	si,offset prolog_b
	call	prolog4
prolog2:
	mov	si,offset prolog_c
prolog4:
	mov	dx,cs:printer	;  (dx) = printer ordinal
prolog6:
	lodsb			;  (al) = next character
	or	al,al		;  Check for end
	je	prolog8		;  Yep
	call 	outchar		;  Write character
	jnc	prolog6		;  Loop
prolog8:
	ret

prolog_a:			;  PostScript program
	db	CTRL_D
	db	'%!PS-Adobe',13,10
	db	'/inch { 72 mul } def',13,10
	db	'/pageheight 11 inch def',13,10
	db	'/top_margin pageheight .4 inch sub def',13,10
	db	'/left_margin .3 inch def',13,10
	db	'/wrap_column 80 def',13,10
	db	'/x left_margin def',13,10
	db	'/y top_margin def',13,10
	db	'/Courier findfont 11 scalefont setfont',13,10
	db	'/lineprint { ',13,10
	db	'x y moveto show',13,10
	db	'/y y 11.4 sub def',13,10
	db	'y .3 inch lt {/y top_margin def showpage} if',13,10
	db	'} def',13,10
	db	'/pageprint {',13,10
	db	'{',13,10
	db	'currentfile 1000 string readline',13,10
	db	'{  ',13,10
	db	0
prolog_b:			;  Code to wrap at given column 
	db	'dup length /nchars exch def',13,10
	db	'{',13,10
	db	'nchars wrap_column le { exit } if',13,10
	db	'dup 0 wrap_column getinterval',13,10
	db	'lineprint',13,10
	db	'/nchars nchars wrap_column sub def',13,10
	db	'wrap_column nchars getinterval',13,10
	db	'} loop',13,10
	db	0
prolog_c:			; And now the rest
	db	'lineprint',13,10
	db	'}',13,10
	db	'{',13,10
	db	'y top_margin ne { showpage} if',13,10
	db	'exit',13,10
	db	'} ifelse',13,10
	db	'} loop',13,10
	db	'} def',13,10
	db	'pageprint',13,10
	db	0

;;	outchar - Output character to printer.
;
;	Entry -	(al) = Character
;		(dx) = Printer ordinal (0 = LPT1:)
;
;	Exit -	Carry set if failed

outchar:
	mov	bl,2		;  (bl) = Retry count
out2:
	mov	ah,0		;  Write character to printer
	int	17H
	mov	bh,al		;  (bh) = character
	call	status		;  Check return status
	jnc	out4		;  Write worked
	sub	bl,1		;  Decrement retry count
	je	out6		;  Limit reached
	mov	al,bh		;  (al) = character
	jmp	out2		;  retry
out4:
	mov	ax,EXIT_SUCCESS
	clc
	ret
out6:
	mov	ah,80H		;  Flag error leaving (al) as is
	stc
	ret


;;	status - Check printer return status.
;
;	Entry -	(ah) = exit status from printer request
;
;	Exit -	Carry clear if O.K.
;		If error (al) indicates error as follows:
;			02 - Timed out
;			09 - Out of paper
;			0A - I/O error

status:
	mov	al,02H
	test	ah,01H		;  Timed out?
	jnz	status2		;  Yep
	mov	al,0AH
	test	ah,08H		;  I/O Error
	jz	status4		;  Nope
	test	ah,20H		;  Out of paper?
	jz	status2		;  Yep
	mov	al,09H
status2:
	stc			;  Flag error
	ret
status4:
	clc			;  Flag O.K.
	ret


;;;	INT17 - Interrupt patch for LPT3 status requests
;
 int17:
	cmp	dx,2		;  Check if LPT3:
	jne	int172		;  Nope
	cmp	ah,2		;  Status Request ?
	jne	int172		;  Nope
	mov	dx,cs:printer	;  Force current printer
int172:
	db	0EAH		;  Far jump
int17_o	dw	?		;  Original int17 offset
int17_s	dw	?		;  Original int17 segment


;;;	Strategy routine.
;
strat:
	mov	cs:[request].uptr.offs,bx
	mov	cs:[request].uptr.segm,es
	retf

;;;	Main interrupt entry.
;
intr:
	push	ds		; Save registers
	push	es
	push	ax
	push	bx
	push	cx
	push	dx
	push	di
	push	si
	mov	ax,cs
	mov	ds,ax
	les	bx,cs:request	; (es:bx) = request block address
	mov	al,es:[bx].req.code ; (al) = request code
	mov	ah,0
	shl	ax,1		; (al) = table offset
	mov	si,ax
	call	cmds[si]	; Call required processor
	les	bx,cs:request	; (es:bx) = request block address
	mov	es:[bx].req.status,ax ; Save returned status
	pop	si		; Restore registers
	pop	di
	pop	dx
	pop	cx
	pop	bx
	pop	ax
	pop	es
	pop	ds
	retf


;;	Initialisation code
;
init:
	mov	ah,9		;  Print banner
	mov	dx,offset init_a
	int	21H

	les	bx,cs:[request]	;  (es:bx) = address of request block
	mov	si,es:[bx].req.count
	mov	ds,es:[bx].req.sector

;	Search for optional parameters.

init2:
	lodsb
	cmp	al,0AH
	je	init10
	cmp	al,'/'		;  Check for parameter
	je	init4
	cmp	al,'-'
	jne	init2		;  Not parameter so ignore

;	Possible parameter

init4:
	lodsb			;  (al) = Key letter
	cmp	al,'W'		;  Wrap request?
	jne	init6		;  Nope
	mov	cs:auto_wrap,80	;  Flag wrap required
	jmp	init2
init6:
	cmp	al,'P'		;  Port number?
	jne	init2		;  Nope
	lodsb			;  (al) = Port number in ascii
	sub	al,'1'		;  Convert to binary
	jl	init8		;  Invalid port number
	cmp	al,1
	ja	init8		;  Invalid

;	Valid port number so save.

	mov	ah,0
	mov	cs:printer,ax
	jmp	init2

;	Invalid printer port specified (must be 1 or 2)

init8:
	mov	ax,cs
	mov	ds,ax
	mov	ah,9		;  Print banner
	mov	dx,offset init_b
	int	21H

;	Print message if auto wrap is selected.

init10:
	mov	ax,cs		;  Restore ds
	mov	ds,ax
	cmp	auto_wrap,0	;  Check if wrap selected
	je	init12		;  Nope
	mov	ah,9
	mov	dx,offset init_c
	int	21H

;	Write out port being used.

init12:
	mov	ax,cs:printer
	add	al,'1'
	mov	cs:init_dx,al
	mov	ah,9
	mov	dx,offset init_d
	int	21H

;	Patch int 17 vector status requests.

	mov	ax,3517H	;  Get current INT17 vector
	int	21H
	mov	int17_s,es	;  And save
	mov	int17_o,bx

	mov	ax,2517H	;  Set new INT17 vector
	mov	dx,offset int17
	int	21H

	mov	ax,init		;  Release initialisation code
	les	bx,cs:[request]	;  (es:bx) = address of request block
	mov	es:[bx].req.adr.uptr.offs,ax
	mov	es:[bx].req.adr.uptr.segm,cs
	mov	ax,EXIT_SUCCESS
	ret

init_a	db	'Installing PostScript device driver on LPT3:',13,10,'$'
init_b	db	'Invalid port specified (must be 1 or 2)',13,10,'$'
init_c	db	'Autowrap selected',13,10,'$'
init_d	db	'Using output port LPT'
init_dx	db	'1:',13,10,'$'

cseg	ends
	end

