/*                        Sprite & Scrolling Engine                         */
/*        Copyright (c) Genesoft 1995-1996.  All rights reserved.           */
/*                                                                          */
/*                             Example Program                              */

#include"sse.h"      // Main header file
#include"ssepal.h"   // Palette functions
#include"ssekeyb.h"  // Keyboard functions
#include"sseutil.h"  // Utility functions
#include"ssepal.h"   // Palette functions

#include<stdio.h>    // File operations
#include<stdlib.h>   //
#include<io.h>       //
#include<fcntl.h>    //
#include<malloc.h>   // Memory allocation
                    
// Our Defines
#define FALSE 0
#define TRUE !FALSE
#define ABS(x) ( x<0 ? -x : x )
#define SQR(x) ( x * x )

#define BUFFER_SIZE 250000
#define NUM_CLOUDS 6

#define CLOUD_X_MIN -50
#define CLOUD_X_MAX (screen_width+50)
#define CLOUD_Y_MIN -50
#define CLOUD_Y_MAX (screen_height+50)
#define CLOUD_X_RANGE (CLOUD_X_MAX-CLOUD_X_MIN)
#define CLOUD_Y_RANGE (CLOUD_Y_MAX-CLOUD_Y_MIN)

#define RAND_DIV (RAND_MAX+1)

// Our Global Variables
char * buffer;       // Using a single buffer (avoiding numerous malloc's)
int left;            // helps to improve performance

sseSpriteHeader * h_body, * h_blade, * h_tail, * h_cloud;
sseTileGraphicInfo * tiles;
sseTileMapInfo * map;

palRegister Wave[2][16];

int esc_hit, pos_x, pos_y;
int scr_max_x, scr_max_y;
int heli_max_x, heli_max_y;
int heli_width, heli_height;
int screen_width, screen_height;

sseSpriteTracking * cloud[NUM_CLOUDS];

// Sin/Cos Pixel-approximations for helicopter directions
char dir_x_table[16] = { 8, 7, 6, 3, 0,-3,-6,-7,-8,-7,-6,-3, 0, 3, 6, 7 };
char dir_y_table[16] = { 0, 3, 6, 7, 8, 7, 6, 3, 0,-3,-6,-7,-8,-7,-6,-3 };

// Internal Helicopter Tracking
typedef struct
{
   int facing, speed, anim, x, y, x_m, y_m;
   int fine_x, fine_x_m, fine_y, fine_y_m;
   sseSpriteTracking * body, * blade, * tail;
} helicopter;
helicopter mainplayer;

// Our Local Function Prototypes
void LoadStuff();
void SetupTables();
void SetupSprites();
char * LoadWholeFile( char * );
void MainLoop();
int HandleKeystrokes();
void UpdatePosition();
void SetupHelicopter( helicopter *, int, int, int );
void UpdateHelicopter( helicopter * );
sseSpriteTracking * NewCloud( int lX, int hX, int lY, int hY );
void UpdateClouds( int sX, int sY );


/***************************************************************************
**                         Main Program Flow                              **
***************************************************************************/

int main( void )
{
   // Load everything
   LoadStuff();

   // Set up our screen and init engine
   sseInitEngine( MODE_320x240x256, tiles, map, 8, 8, 0 );
   screen_width = 320;
   screen_height = 240;

   // Calculate the maximum x and y values we can scroll to
   scr_max_x = map->Width * tiles->Width;
   scr_max_y = map->Height * tiles->Height;

   // Get our sprites ready
   SetupSprites();

   // Install keyboard handler
   keyInstallHandler();

   // Execute our main loop
   MainLoop();

   // Restore old keyboard handler
   keyRestoreHandler();

   // Return to text mode
   sseRestoreVideoMode();

   return EXIT_SUCCESS;
}

/***************************************************************************
**                 Loading and Initializing Functions                     **
***************************************************************************/

void LoadStuff()
{
   // Set up a buffer for our data
   if( ( buffer = (char*) malloc( BUFFER_SIZE ) ) == NULL )
   {
      printf( "\n Error Allocating Work Buffer!\n\n" );
      exit( EXIT_FAILURE );
   }
   left = BUFFER_SIZE;

   // Load our tiles
   tiles = (sseTileGraphicInfo*) LoadWholeFile( "TILES.DAT" );
   
   // Create copies of wave-section of the palette
   memCopy( Wave[0], tiles->Palette+120, 16 * sizeof(palRegister) );
   palCycle( Wave[0], 1, 16 );
   palBlendPalettes( Wave[1], Wave[0], 32, tiles->Palette+120, 16 );

   // Load our map
   map = (sseTileMapInfo*) LoadWholeFile( "WORLD.MAP" );

   // Load our sprites
   h_body = (sseSpriteHeader*) LoadWholeFile( "HELIBODY.SSP" );
   h_tail = (sseSpriteHeader*) LoadWholeFile( "HELITAIL.SSP" );
   h_blade = (sseSpriteHeader*) LoadWholeFile( "HELIBLAD.SSP" );
   h_cloud = (sseSpriteHeader*) LoadWholeFile( "CLOUD.SSP" );
}

void SetupSprites()
{
   sseSpriteGraphicInfo * g;
   int k;
   
   // Register our sprite data
   sseRegisterSprite( h_body );
   sseRegisterSprite( h_tail );
   sseRegisterSprite( h_blade );
   sseRegisterSprite( h_cloud );

   // Center our cloud's pivot coordinates
   g = h_cloud->Frame;
   g->PivotX = g->Width / 2;
   g->PivotY = g->Height / 2;
   
   // Create the initial clouds
   for( k=0; k<NUM_CLOUDS; k++ )
      cloud[k] = NewCloud( CLOUD_X_MIN, CLOUD_X_MAX, CLOUD_Y_MIN, CLOUD_Y_MAX);
   
   // Calculations for future use
   heli_width = h_body->Frame[0].Width;
   heli_height = h_body->Frame[0].Height;
   heli_max_x = map->Width * tiles->Width - heli_width;
   heli_max_y = map->Height * tiles->Height - heli_height;

   // Set up internal tracking and sprite tracking
   SetupHelicopter( & mainplayer, heli_max_x/2, heli_max_y/2, 0 );
   pos_x = mainplayer.x - (screen_width - heli_width) /2;
   pos_y = mainplayer.y - (screen_height - heli_height) /2;
}

char * LoadWholeFile( char * name )
{
   int fh, fs;
   void LoadError( char * );
   void ReadError( char * );
   void IMessedUp( char * );

   if( ( fh = open( name, O_RDONLY | O_BINARY ) ) == -1 )
      LoadError( name );
   if( ( fs = read( fh, buffer, left ) ) == -1 )
      ReadError( name );
   if( fs == left )
      IMessedUp( name );
   fs = (fs + 3) & 0xFFFFFFFC;   // Round to nearest DWORD boundry
   buffer += fs;
   left -= fs;
   close( fh );
   return buffer - fs;
}

void LoadError( char * str )
{
   printf( "\n Error Loading \"%s\"!\n\n", str );
   exit( 1 );
}
void ReadError( char * str )
{
   printf( "\n Error Reading File \"%s\"!\n\n", str );
   exit( 1 );
}
void IMessedUp( char * str )
{
   printf( "\n Oops!  I didn't allocate enough memory for \"%s\"!\n\n", str );
   exit( 1 );
}


/***************************************************************************
**                           Main Game Loop                               **
***************************************************************************/

void MainLoop()
{
   palRegister new_pal[256];
   int darkness = 64, fade_dir = -2;
   int wnum = 0;

   // Zero initial palette
   memByteFill( new_pal, 0, 256 * sizeof( palRegister ) );
   
   while( darkness <= 64 )  // Continue main loop until screen has faded-out
   {
      // Set new palette
      if( fade_dir )         // If we're fading in/out set whole palette
         palSetActive( new_pal, 0, 256 );
      else                   // otherwise, just update the wave-portion (120-136)
         palSetActive( tiles->Palette+120, 120, 16 );

      // Read user input and if exiting, start fade-out
      if( HandleKeystrokes() ) fade_dir = 2;

      // Update our sprites' locations and frames
      UpdateHelicopter( & mainplayer );
      
      // Find our new screen position
      UpdatePosition();
      
      // Move the clouds
      UpdateClouds( mainplayer.x_m, mainplayer.y_m );

      // Create new palette
      palCycle( Wave[wnum], -1, 16 );
      memCopy( tiles->Palette+120, Wave[wnum], 16*sizeof(palRegister));
      wnum ^= 1;
      if( fade_dir )
      {
         palDarken( new_pal, tiles->Palette, darkness, 256 );
         darkness += fade_dir;
         if( darkness < 0 ) { fade_dir = 0; darkness = 0; }
      }

      // Update the screen
      sseUpdateScreen( pos_x, pos_y );
   }
}

int HandleKeystrokes()
{
   char up, left, right;
   
   if( ESC_PRESSED ) return TRUE;
   else
   {
      // Get all of our direction keystrokes at once
      up = UP_PRESSED;
      left = LEFT_PRESSED;
      right = RIGHT_PRESSED;

      // Adjust movement direction by keystrokes
      if( up ) mainplayer.speed++;
      else mainplayer.speed--;
      if( left && ! right ) mainplayer.facing--;
      else if( right && ! left ) mainplayer.facing++;

      return FALSE;
   }
}

void UpdatePosition()
{
   // Find map coordinates relative to mainplayer's position
   pos_x = mainplayer.x - (screen_width - heli_width) /2;
   pos_y = mainplayer.y - (screen_height - heli_height) /2;

   // Check if we've wrapped around the left or top edges of the map.
   //   since mainplayer.x,y always > pos_x,y , pos_x,y < scr_max_x,y
   if( pos_x < 0 ) pos_x += scr_max_x;
   if( pos_y < 0 ) pos_y += scr_max_y;
}

/***************************************************************************
**                        Helicopter Functions                            **
***************************************************************************/

#define FINE_SCALE 256

void SetupHelicopter( helicopter * h, int x, int y, int f )
{
   h->facing = f;
   h->speed = 0;
   h->anim = 0;
   h->fine_x = (x * FINE_SCALE) + (FINE_SCALE/2);
   h->fine_y = (y * FINE_SCALE) + (FINE_SCALE/2);
   h->fine_x_m = 0;
   h->fine_y_m = 0;
   h->x = h->fine_x / FINE_SCALE;
   h->y = h->fine_y / FINE_SCALE;
   x = ( screen_width - heli_width ) /2;
   y = ( screen_height - heli_height ) /2;
   h->body = sseInitSprite( h_body, x, y, 0, 1 ); 
   h->tail = sseInitSprite( h_tail, x, y, 0, 2 );
   h->blade = sseInitSprite( h_blade, x, y, 0, 3 );
}

void UpdateHelicopter( helicopter * h )
{
   int dir, t;

   // Wrap facing value if below 0 or above 63   48
   //                                          32 + 00
   //                                             16
   h->facing = h->facing & 63;

   // Take 1/4 of facing value for actual direction  12
   //                                              08 + 00
   //                                                 04
   dir = h->facing >> 2;

   // Keep speed value between 0 and 16
   if(h->speed < 0) h->speed = 0;
   if(h->speed >= 16)
   {
      h->speed = 16;
      
      // If momentum has maxed out, force movement to the exact pixel value
      // to avoid jerky movement
      t = dir_x_table[dir] < 0 ? -15 : 15;
      if( dir_x_table[dir] == (h->fine_x_m+t) / FINE_SCALE )
         h->fine_x_m = dir_x_table[dir] * FINE_SCALE;
      t = dir_y_table[dir] < 0 ? -15 : 15;
      if( dir_y_table[dir] == (h->fine_y_m+t) / FINE_SCALE )
         h->fine_y_m = dir_y_table[dir] * FINE_SCALE;
   }

   // Decrease momentum by 1/16th, rounding towards 0
   if( h->fine_x_m & 15 ) h->fine_x_m += h->fine_x_m > 0 ? -1 : 1;
   h->fine_x_m -= h->fine_x_m / 16;
   if( h->fine_y_m & 15 ) h->fine_y_m += h->fine_y_m > 0 ? -1 : 1;
   h->fine_y_m -= h->fine_y_m / 16;

   // Increase momentum by our direction times our speed
   h->fine_x_m += dir_x_table[ dir ] * h->speed;
   h->fine_y_m += dir_y_table[ dir ] * h->speed;

   // Adjust position by our momentum and find pixel movement
   h->x_m = h->fine_x / FINE_SCALE;
   h->y_m = h->fine_y / FINE_SCALE;
   h->fine_x += h->fine_x_m;
   h->fine_y += h->fine_y_m;
   h->x_m -= h->fine_x / FINE_SCALE;
   h->y_m -= h->fine_y / FINE_SCALE;
   
   // Check for wrapping around the map edges
   if( h->fine_x < 0 )
      h->fine_x += scr_max_x * FINE_SCALE;
   else if( h->fine_x > scr_max_x * FINE_SCALE )
      h->fine_x -= scr_max_x * FINE_SCALE;
   if( h->fine_y < 0 )
      h->fine_y += scr_max_y * FINE_SCALE;
   else if( h->fine_y > scr_max_y * FINE_SCALE )
      h->fine_y -= scr_max_y * FINE_SCALE;

   // Find the actual pixel position for the helicopter
   h->x = h->fine_x / FINE_SCALE;
   h->y = h->fine_y / FINE_SCALE;

   // Set sprites' frame numbers by anim counter and direction
   h->body->Frame = dir;
   if( ++h->blade->Frame >= h->blade->Header->NumFrames ) h->blade->Frame = 0;
   if( ++h->anim > 3 ) h->anim = 0;
   if( h->anim == 3 ) h->tail->Frame = 1 * 16 + dir;
   else               h->tail->Frame = h->anim * 16 + dir;
}

/***************************************************************************
**                           Cloud Functions                              **
***************************************************************************/

int RandomInt( int min, int max )
{
   return (rand() / (RAND_DIV/(max-min)) + min);
}

sseSpriteTracking * NewCloud( int lX, int hX, int lY, int hY )
{
   return sseInitSprite( h_cloud, RandomInt( lX, hX), RandomInt( lY, hY ), 0,5);
}

void UpdateClouds( int sX, int sY )
{
   int count, k, x, y;
   static int xrem = 0, yrem = 0;
   sseSpriteTracking ** c;
   
   // Keep track of x,y pixel remainders for movement slower than 1 pexel/update
   // Move clouds ( 1.5 * sX, 1.5 * sY - 0.25 )
   xrem += (sX << 1);
   yrem += (sY << 1) -1;
   sX += xrem >> 2;
   sY += yrem >> 2;
   xrem &= 3;
   yrem &= 3;
   
   // Update each cloud
   for( count=0, k=0, c=cloud; k<NUM_CLOUDS; k++, c++ )
   {
      if( !*c ) continue;  // No cloud here
      (*c)->X += sX;
      (*c)->Y += sY;
      // Check if it has gone far enough off-screen to be disabled
      if( (*c)->X < CLOUD_X_MIN || (*c)->Y < CLOUD_Y_MIN || (*c)->X > CLOUD_X_MAX ||
          (*c)->Y > CLOUD_Y_MAX )
      {
         sseDisableSprite( *c );
         *c = 0;
      }
      else count++;
   }

   // If we've scrolled anywhere and have less than the max # of clouds,
   // we may want to create another
   if( (sX || sY) && rand() * 30 / RAND_DIV < NUM_CLOUDS-count )
   {
      // find a free position
      c=cloud;
      while( *c ) c++;
      // choose a location from the area being scrolled in
      k = rand() * ( ABS(sX) * CLOUD_Y_RANGE + ABS(sY) * CLOUD_X_RANGE - ABS(sX*sY) ) /RAND_DIV;
      if( k < ABS(sX) * CLOUD_Y_RANGE )
      {
         // Cloud is being created on the top or bottom of the screen
         x = sX < 0 ? CLOUD_X_MAX - sX + (k%sX) : CLOUD_X_MIN + (k%(-sX));
         y = CLOUD_Y_MIN + (k/ABS(sX));
      }
      else
      {
         // Cloud is being created on the left or right edge of the screen
         k -= ABS(sX) * CLOUD_Y_RANGE;
         x = CLOUD_X_MIN + (k/ABS(sY));
         y = sY < 0 ? CLOUD_Y_MAX - sY + (k%sY) : CLOUD_Y_MIN + (k%(-sY));
      }
      *c = sseInitSprite( h_cloud, x, y, 0, 5 );
   }
}
