/*****************************************************************************
*                           BC 3.2 LIBRARY                                   *
*                         SB16 ROUTINES V1.0                                 *
*                              SB16.CPP                                      *
*                                                                            *
*  This product is copyrighted (C) 1996 by Henk Thomassen. Permission is     *
*  granted for non-commercial use and distribution.                          *
******************************************************************************
* REMARK: All sb16 mixer routines in this library are ALLWAYS setting the    *
*         right and left channel in a equal state. ( volume, in- and output  *
*         settings)                                                          *
*                                                                            *
*	  The sbinit() routine is an adaptation from Ethan Brodsky's         *
*	  init routine. (free ware 1995)                                     *
*                                                                            *
*          v1.0 Jan 1996                                                     *
*                                                                            *
******************************************************************************
* If you have any questions or remarks please contact me                     *
* Henk Thomassen, thomass@IAEhv.nl                                           *
*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <mem.h>
#include <malloc.h>
#include <conio.h>
#include <ctype.h>
#include <dos.h>
#include <string.h>
#include "sb16.h"

#define TRUE    1
#define FALSE   (!TRUE)
#define hi(x) ((x) >> 8)
#define lo(x) ((x) & 0xFF)

int far *raw_16;
char far *raw_8;
int sb_baseio;
int sb_irq;
int sb_dma;
int sb_dma16;
int sb_dspversion;
int sb16_present=FALSE;

static unsigned char far *data;
static unsigned char aligned_physical;

static int mix_register;               //02x04
static int mix_data;                   //02x05

static int dsp_resetport;              //02x06
static int dsp_readport;               //02x0a
static int dsp_writeport;              //02x0c
static int dsp_bufferstat;             //02x0c
static int dsp_timerintclear;          //02x0c
static int dsp_pollport;               //02x0e
static int dsp_ackport8;               //02x0e
static int dsp_ackport16;              //02x0f

static int pic_rotateport;
static int pic_maskport;
static int irq_startmask;
static int irq_stopmask;
static int irq_intvector;

static int dma8_maskport;
static int dma8_clrptrport;
static int dma8_modeport;
static int dma8_addrport;
static int dma8_countport;
static int dma8_pageport;
static int dma8_startmask;
static int dma8_stopmask;

static int dma8_singleplay;
static int dma8_autoplay;
static int dma8_singlerec;
static int dma8_autorec;

static int dma16_maskport;
static int dma16_clrptrport;
static int dma16_modeport;
static int dma16_addrport;
static int dma16_countport;
static int dma16_pageport;
static int dma16_startmask;
static int dma16_stopmask;

static int dma16_singleplay;
static int dma16_autoplay;
static int dma16_singlerec;
static int dma16_autorec;


static int get_setting(char *str, char id, int *value);
static void dspwrite(unsigned char value);
static unsigned char read_dsp(void);
static int reset_dsp(void);


/****************************************************************************
* INIT:                                                                     *
****************************************************************************/
static int get_setting(char *str, char id, int &value)
  {
    char *paramstart;
    char buf1[128];
    char buf2[128];

    strcpy(buf1, str);
    if (strchr(buf1, id) != NULL)
      {
	paramstart = strchr(buf1, id) + 1;

	if (strchr(paramstart, ' ') != NULL)
	  *(strchr(paramstart, ' ')) = '\0';

	strcpy(buf2, "0x");
	strcat(buf2, paramstart);

	value = strtoul(buf2, NULL, 0);
	return(TRUE);
      }
    else
      {
	value = 0;
	return(FALSE);
      }
  }

/****************************************************************************
* INIT:                                                                     *
****************************************************************************/
int detect_blaster()
  {
    char blaster[128];

    if (getenv("BLASTER") == NULL)
      {
	sb_baseio = 0x220;
	sb_irq    = 5;
	sb_dma    = 1;
	sb_dma16  = 5;
	return (FALSE);
      }

    strupr(strcpy(blaster, getenv("BLASTER")));

    if (!get_setting(blaster, 'A', sb_baseio)) return(FALSE);
    if (!get_setting(blaster, 'I', sb_irq))    return(FALSE);
    if (!get_setting(blaster, 'D', sb_dma))    return(FALSE);
    if (!get_setting(blaster, 'H', sb_dma16))  return(FALSE);

    return(TRUE);
}


/****************************************************************************
* INIT/DMA:                                                                 *
****************************************************************************/
static void dspwrite(unsigned char value)
  {
    while ((inp(dsp_writeport) & 0x80));
    outp(dsp_writeport, value);
  }

/****************************************************************************
* INIT/DMA:                                                                 *
****************************************************************************/
static unsigned char read_dsp(void)
  {
    while (!(inp(dsp_pollport) & 0x80));
    return(inp(dsp_readport));
  }

/****************************************************************************
*  INIT:                                                                    *
****************************************************************************/
static int reset_dsp(void)
  {
    int i;

    outp(dsp_resetport, 1);
    delay(1);                            // 1 millisecond
    outp(dsp_resetport, 0);

    i = 100;
    while ((i-- > 0) && (read_dsp() != 0xAA));

    return(i > 0);
  }

/****************************************************************************
*  INIT:                                                                    *
****************************************************************************/
int sbinit(int baseio, int irq, int dma, int dma16)
  {
    // Sound card IO ports
    mix_register     = baseio + 0x004;
    mix_data         = baseio + 0x005;

    dsp_resetport    = baseio + 0x006;
    dsp_readport     = baseio + 0x00A;
    dsp_writeport    = baseio + 0x00C;
    dsp_bufferstat   = baseio + 0x00C;
    dsp_timerintclear= baseio + 0x00D;
    dsp_pollport     = baseio + 0x00E;
    dsp_ackport8     = baseio + 0x00E;
    dsp_ackport16    = baseio + 0x00F;

    sb16_present=FALSE;
    // Reset DSP, get version, choose output mode
    if (!reset_dsp()) return(FALSE);

    // Get DSP version number
    dspwrite(0xE1);
    sb_dspversion = read_dsp();  sb_dspversion += read_dsp() / 100.0;
    if(sb_dspversion<4) return(FALSE);

    sb16_present=TRUE;
   // Compute interrupt controller ports and parameters
    if (irq < 8)
      { // PIC1
	irq_intvector  = 0x08 + irq;
	pic_rotateport = 0x20;
	pic_maskport   = 0x21;
      }
    else
      { // PIC2
	irq_intvector  = 0x70 + irq-8;
	pic_rotateport = 0xA0;
	pic_maskport   = 0xA1;
      }
    irq_stopmask  = 1 << (irq % 8);
    irq_startmask = ~irq_stopmask;

    // Compute DMA controller ports and parameters

    // Sixteen bit
    dma16_maskport   = 0xD4;
    dma16_clrptrport = 0xD8;
    dma16_modeport   = 0xD6;
    dma16_addrport   = 0xC0 + 4*(dma16-4);
    dma16_countport  = 0xC2 + 4*(dma16-4);
    switch (dma16)
    {
      case 5: dma16_pageport = 0x8B; break;
      case 6: dma16_pageport = 0x89; break;
      case 7: dma16_pageport = 0x8A; break;
    }
    dma16_stopmask   = dma16-4 + 0x04; // 0000 01xx //
    dma16_startmask  = dma16-4 + 0x00; // 0000 00xx //

    dma16_singleplay = dma16-4 + 0x48; // 0100 10xx //
    dma16_autoplay   = dma16-4 + 0x58; // 0101 10xx //
    dma16_singlerec  = dma16-4 + 0x44; // 0100 01xx //
    dma16_autorec    = dma16-4 + 0x54; // 0101 01xx //

    // Eight bit
    dma8_maskport   = 0x0A;
    dma8_clrptrport = 0x0C;
    dma8_modeport   = 0x0B;
    dma8_addrport   = 0x00 + 2*dma;
    dma8_countport  = 0x01 + 2*dma;
    switch (dma)
    {
      case 0: dma8_pageport = 0x87; break;
      case 1: dma8_pageport = 0x83; break;
      case 2: dma8_pageport = 0x81; break;
      case 3: dma8_pageport = 0x82; break;
    }
    dma8_stopmask  = dma + 0x04;      // 0000 01xx //
    dma8_startmask = dma + 0x00;      // 0000 00xx //

    dma8_singleplay = dma + 0x48;     // 0100 10xx //
    dma8_autoplay   = dma + 0x58;     // 0101 10xx //
    dma8_singlerec  = dma + 0x44;     // 0100 01xx //
    dma8_autorec    = dma + 0x54;     // 0101 01xx //

    return(TRUE);
}

/****************************************************************************
* BUFFER:                                                                   *
****************************************************************************/
int sbmalloc ( void )
{
    unsigned long physical;
    data = (unsigned char far *) farmalloc(80000L);

    if(data==NULL) return(FALSE);

    physical=((unsigned long)FP_OFF(data))+(((unsigned long)FP_SEG(data))<<4);
    physical+=0x0FFFFL;
    physical&=0xF0000L;
    aligned_physical=(physical>>16)&15;
    raw_16=(int far *) MK_FP(((unsigned short)aligned_physical<<12)&0xF000,0);
    raw_8=(unsigned char far *) raw_16;

    return(TRUE);
}

/****************************************************************************
* BUFFER:                                                                   *
****************************************************************************/
void sbfree()
{
  farfree(data);
}

/****************************************************************************
* SAMPLE FREQ:                                                              *
****************************************************************************/
void sbsettc (unsigned int f)
{
  if(sb16_present){
    inportb(dsp_pollport);
    dspwrite(0x42);
    dspwrite(hi(f));
    dspwrite(lo(f));
  }
}

/****************************************************************************
*  8 BIT DMA:                                                               *
****************************************************************************/
void sbrec8 ( unsigned short len )
{
  if(sb16_present){
    len--;
    outportb(dma8_maskport,dma8_stopmask);
    outportb(dma8_clrptrport,0x00);
    outportb(dma8_modeport,dma8_singlerec);
    outportb(dma8_addrport,0);
    outportb(dma8_addrport,0);
    outportb(dma8_pageport,aligned_physical);
    outportb(dma8_countport,lo(len));
    outportb(dma8_countport,hi(len));
    outportb(dma8_maskport,dma8_startmask);

    dspwrite(0xC8);
    dspwrite(0x10);
    dspwrite(lo(len));
    dspwrite(hi(len));
  }
}

/****************************************************************************
*  8 BIT DMA:                                                               *
****************************************************************************/
unsigned short dmacount8 ( void )
{
    unsigned short x=0xffff;
    if(sb16_present){
      x=inportb(dma8_countport);
      x|=inportb(dma8_countport)<<8;
      if(x==0xFFFF) inportb(dsp_ackport8);
    }
    return(x);
}

/****************************************************************************
*  16 BIT DMA:                                                              *
****************************************************************************/
void sbrec16 ( unsigned short len )
{
  unsigned int offset;

  if(sb16_present){
    offset = ((aligned_physical * 65536L) / 2) % 65536;
    len--;
    outportb(dma16_maskport,dma16_stopmask);
    outportb(dma16_clrptrport,0x00);
    outportb(dma16_modeport,dma16_singlerec);
    outportb(dma16_addrport,lo(offset));
    outportb(dma16_addrport,hi(offset));
    outportb(dma16_pageport,aligned_physical);
    outportb(dma16_countport,lo(len));
    outportb(dma16_countport,hi(len));
    outportb(dma16_maskport,dma16_startmask);

    dspwrite(0xB8);
    dspwrite(0x10);
    dspwrite(lo(len));
    dspwrite(hi(len));
  }
}

/****************************************************************************
*  16 BIT DMA:                                                              *
****************************************************************************/
unsigned short dmacount16 ( void )
{
    unsigned short x=0xffff;
    if(sb16_present){
      x=inportb(dma16_countport);
      x|=inportb(dma16_countport)<<8;
      if(x==0xFFFF) inportb(dsp_ackport16);
    }
    return(x);
}

/****************************************************************************
*  MIXER:                                                                   *
****************************************************************************/
unsigned char SBgetmixreg(unsigned char r)
{
  if(sb16_present){
    outportb(mix_register,r);
    return(inportb(mix_data));
  }
  else
    return(0);
}

/****************************************************************************
* MIXER:  							            *
****************************************************************************/
void SBsetmixreg(unsigned char r, unsigned char d)
{
  if(sb16_present){
    outportb(mix_register,r);
    outportb(mix_data,d);
  }
}

/****************************************************************************
* MIXER:  	         						    *
****************************************************************************/
int getVolume(int n)
{
  int mregr,mregl;
  unsigned char v=0;

  switch(n)
  {
    case 0 : mregl=0x30; mregr=0x31; break;
    case 1 : mregl=0x38; mregr=0x39; break;
    case 2 : mregl=0x36; mregr=0x37; break;
    case 3 : mregl=0x3a; mregr=0x3a; break;
    default: mregl=0;    mregr=0;
  }
  if(mregl!=0)
  {
    v=SBgetmixreg(mregr);
    SBsetmixreg(mregl,v); // set volume right == left
  }
  return(31-(v>>3));              // scale 0..31
}

/****************************************************************************
* MIXER:                                                                    *
****************************************************************************/
void setVolume(int n, unsigned char v)
{
  int mregr,mregl;
  switch(n)
  {
    case 0 : mregl=0x30; mregr=0x31; break;
    case 1 : mregl=0x38; mregr=0x39; break;
    case 2 : mregl=0x36; mregr=0x37; break;
    case 3 : mregl=0x3a; mregr=0x3a; break;
    default: mregl=0;    mregr=0;
  }
  if(mregl!=0)
  {
    SBsetmixreg(mregr,(31-v)<<3);  // set left and right
    SBsetmixreg(mregl,(31-v)<<3);
  }
}

/****************************************************************************
* MIXER:      								    *
****************************************************************************/
void setSBswitch(int n, int on)
{
  unsigned char r,swi,swo,ampi,ampo;

  swi=SBgetmixreg(0x03e);    // swi  = input control
  ampi=SBgetmixreg(0x040);   // ampi = input gain control
  swo=SBgetmixreg(0x03c);    // swo  = output control
  ampo=SBgetmixreg(0x041);   // ampo = outout gain control

  switch(n)
  {
  case 2: swi=(on) ? (swi|0x10) : (swi&(~0x10)); break;
  case 3: swi=(on) ? (swi|0x08) : (swi&(~0x08)); break;
  case 6: swi=(on) ? (swi|0x04) : (swi&(~0x04)); break;
  case 7: swi=(on) ? (swi|0x02) : (swi&(~0x02)); break;
  case 9: swi=(on) ? (swi|0x01) : (swi&(~0x01)); break;

  case 0: swo=(on) ? (swo|0x10) : (swo&(~0x10)); break;
  case 1: swo=(on) ? (swo|0x08) : (swo&(~0x08)); break;
  case 4: swo=(on) ? (swo|0x04) : (swo&(~0x04)); break;
  case 5: swo=(on) ? (swo|0x02) : (swo&(~0x02)); break;
  case 8: swo=(on) ? (swo|0x01) : (swo&(~0x01)); break;

  case 10: ampo=on<<6; break;
  case 11: ampi=on<<6; break;
  }

  SBsetmixreg(0x03e,swi);   // swi = input control
  SBsetmixreg(0x03d,swi);   // left == right;

  SBsetmixreg(0x040,ampi);  // ampi = input gain control
  SBsetmixreg(0x03f,ampi);  // left == right;

  SBsetmixreg(0x03c,swo);   // swo = output control

  SBsetmixreg(0x041,ampo);  // ampo = outout gain control
  SBsetmixreg(0x042,ampo);  // left == right;
}

/****************************************************************************
* MIXER:								    *
****************************************************************************/
int getSBswitch(int n)
{
  unsigned char r,swi,swo,ampi,ampo;

  swi=SBgetmixreg(0x03e);    // swi  = input control
  ampi=SBgetmixreg(0x040);   // ampi = input gain control
  swo=SBgetmixreg(0x03c);    // swo  = output control
  ampo=SBgetmixreg(0x041);   // ampo = outout gain control

  SBsetmixreg(0x03d,swi);    // left == right;
  SBsetmixreg(0x03f,ampi);
  SBsetmixreg(0x042,ampo);

  switch(n)
  {
  case 2: r=swi&0x10; break;
  case 3: r=swi&0x08; break;
  case 6: r=swi&0x04; break;
  case 7: r=swi&0x02; break;
  case 9: r=swi&0x01; break;

  case 0: r=swo&0x10; break;
  case 1: r=swo&0x08; break;
  case 4: r=swo&0x04; break;
  case 5: r=swo&0x02; break;
  case 8: r=swo&0x01; break;

  case 10: r=ampo>>6; break;
  case 11: r=ampi>>6; break;
  }
  return(r);
}

/********************************* END ************************************/
